/* full screen disk sector editor */

/*
author:
   raymond l. zarling
   dept. computer science
   cal state univ, stanislaus
   turlock, ca 95380

written for:
   commodore 64 computer
compiler:
   c-power compiler v2.4,
      pro-line software, ltd.
date:
   august, 1985
*/

#include <stdio.h>

/* the following are useful for clarity
and to conserve variable space.
unfortunately, c-power 2.4 does not
permit formal parameters
that are one byte long, so
these are declared "unsigned" with
comments declaring the real type. */

#define BYTE char
#define BOOL char
#define VOID int

#define TRUE 1
#define FALSE 0

/* c64 memory locations */

#define JCLOCK (char *)0x00a2
#define TCOL (char *)0x0286
#define ENABL (char *)0x02a1
#define SID 0xd400

/* colors: */

#define BACKCOL 12
#define BORDCOL 12
#define COMPCOL 0
#define USERCOL 1

#define SETCOMP *TCOL = COMPCOL
#define SETUSER *TCOL = USERCOL

/* special characters */

#define CLRS 147
#define HOME 19
#define LC 14
#define CB 157
#define CD 17

/* functions which return a char length
result may not be explicity declared
under c-power v2.4:

VOID printv();
char *cvt();
int eval();
VOID mod();
BOOL dopen();
VOID dclose();
BYTE diskerr();
VOID instr1();
VOID instr2();
VOID dsect();
VOID display();
BOOL badts();
BOOL device();
VOID stall();
VOID lineout();
char toascii();
VOID tone();
*/

static BYTE tr, se;
static char mode;
static BOOL rs232;
static FILE outf;
static char image[41];

static char back9[]
       = { CB, CB, CB, CB, CB, CB, CB, CB, CB, 0 };

static char sector[256];
static char line[81];

main()
{
   static BYTE offset, newbaud;
   register char *lp, *bp, ch;
   register BOOL update, more;
   static char baud[] = { 6, 0 };

   outf = stdout;
   rs232 = 0;

   *(char *)0xd020 = BORDCOL;
   *(char *)0xd021 = BACKCOL;
   putchar( LC );

   *sector = 18; sector[ 1 ] = 1;
   mode = 'A';

   for (;;)
   {
      tr = *sector;
      se = sector[ 1 ];
      if ( badts () )
      {
         tr = 18; se = 1;
      }


      /* Ask for Next Sector to Display: */

      instr1();
      printf ( "Track, Sector, mode? %02d, %02d, %c%s",
         tr, se, mode, back9 );

      SETUSER;
      gets ( line );
      if ( feof ( stdin ) ) break;
      lp = cvt ( line, &tr, 10 );
      lp = cvt ( lp, &se, 10 );
      while ( *lp ) lp++;
      ch = (*--lp) | 0x80;
      if ( ch == 'H' || ch == 'A' )
         mode = ch;
      if ( badts () )
         break;


      /* Display the Sector: */

      if ( dopen() ) continue;
      fprintf ( 9, "u1:8 0 %d %d\n", tr, se );
      if ( diskerr( "Reading Disk" ) ) continue;

      putchar ( CLRS );

      offset = 0;
      bp = sector;
      do
         *bp++ = getc ( 8 );
      while ( ++offset );

      dclose ();

      display ();


      /* Ask for Modifications: */

      update = 0;
      more = 1;
      do
      {
         printf ( "%c%cx%c", HOME, CD, CB );
         SETUSER;
         *line = '\0';
         gets ( line );
         if ( feof ( stdin ) ) exit();

         ch = *line;
         switch ( ch )
         {
         case ':':
            lp = cvt ( line, &offset, 16 );
            mod ( lp, offset );
            break;
         case 'p':
         case 'P':
            if ( device ( 4, "\021" ) )
               dsect ();
            device ( 3 );
            break;
         case 'r':
         case 'R':
            cvt ( line, &newbaud, 10 );
            if (newbaud) *baud = newbaud;
            device ( 2, baud );
            dsect ();
            device ( 3 );
            break;
         case '-':
            lp = cvt ( line, &tr, 10 );
            cvt ( lp, &se, 10 );
            if ( badts () ) abort();
            break;
         case 'a':
         case 'A':
         case 'h':
         case 'H':
            mode = ch | 0x80;
            putchar ( HOME );
            display ();
            break;
         case 'w':
         case 'W':
            update = 1;
         case 'x':
         case 'X':
            more = 0;
            break;
         }
      } while ( more );

      if ( update )
      {
         if ( dopen() ) continue;
         fprintf ( 9, "b-p:8 0\n" );
         offset = 0;
         bp = sector;
         do
         {
            putc ( *bp++, 8 );
         }  while ( ++offset );
         fprintf ( 9, "u2:8 0 %d %d\n", tr, se );
         if ( diskerr( "Writing Disk" ) ) continue;
         dclose();
      }

   } /* for(;;) */

   SETCOMP;

} /* end of main program */

VOID printv ( s, ch )
char *s, ch;
/* convert ch to a string at s, according
to current mode. */
{
   register char *fmt;

   if ( (mode == 'A') &&
      ( isprint ( ch ) || isupper ( ch ) ) )
      fmt = "'%c";
   else
      fmt = "%02x";
   sprintf ( s, fmt, ch );
}

char *cvt ( p, v, b )
char *p;
BYTE *v, b;
/* convert the next base b number found
beginning at p to a byte at *v.  return
zero if no base b number found. */
{
   register int d;
   register BYTE r;

   while ( *p && eval( *p, b ) < 0 )
      p++;
   r = 0;
   for ( ; *p && (d = eval (*p, b)) >= 0; p++ )
      r = r * b + d;
   *v = r;
   return p;
}

int eval ( ch, b )
unsigned /* char */ ch;
BYTE b;
/* evaluate the cbmascii char ch in the
given numbering system b (<=16).  return
the result, or -1 upon error. */
{
   register int result;
   static char digits[] = "0123456789abcdef";

   ch = ch & 0x7f;
   result = b;
   while ( --result >= 0 )
      if ( digits[ result ] == ch ) break;
   return result;
}

VOID ldisp ( offset )
BYTE offset;
/* display one line of the sector at the
given offset on the current device. */
{
   register BYTE j;
   register char *bp, *s;

   SETCOMP;
   bp = sector + offset;
   sprintf ( image, ":%02x", offset );
   s = image + 3;
   for (j=0; j<16; j++)
   {
      if ( !(j & 3) ) *s++ = ' ';
      printv ( s, *bp++ );
      s += 2;
   }
   *s = '\0';
   lineout ( image );
}

VOID mod ( line, o )
char *line;
BYTE o;
/* modify a line of data at given offset
into the sector.  if error, do no modification,
and redisplay entire sector */
{
   register BYTE i, j, result, state, offset;
   register char *lp, ch;
   register int e;
   auto char data[40];

#define ERRSTATE 0
#define STRT 1
#define ONEDIGIT 2
#define GOTQUOTE 3

   offset = o;

   i = 0;
   if ( *line ) state = STRT;
   else state = ERRSTATE;
   while ( state && (ch = *++line) )
   {
      e = eval ( ch, 16 );
      switch ( state )
      {
      case STRT:
         if ( ch == ' ' ) continue;
         if ( ch == '\'' ) state = GOTQUOTE;
         else if ( e >= 0 )
         {
            result = e;
            state = ONEDIGIT;
         }
         else state = ERRSTATE;
         break;
      case ONEDIGIT:
         state = STRT;
         if ( e >= 0 )
            data[i++] = (result<<4) + e;
         else
            state = ERRSTATE;
         break;
      case GOTQUOTE:
         data [i++] = ch;
         state = STRT;
      }
   }

   if ( state == GOTQUOTE )
   {
      data [ i++ ] = ' ';
      state = STRT;
   }
   if ( state != STRT )
   {
      tone ();
      display ();
   }
   else
   {
      for ( j=0; j<i && offset<256; j++)
         sector[ offset++ ] = data [ j ];
   }
}

BOOL dopen()
/* return TRUE if error opening disk */
{
   open( 9, 8, 15, "i0:" );
   if ( diskerr ( "Initializing Disk" ) )
      return TRUE;

   open ( 8, 8, 8, "#1" );
   if ( diskerr ( "Opening Data Channel" ) )
      return TRUE;

   return FALSE;
}

VOID dclose()
{
/* i know you should use "close" when
   the file has been opened with "open",
   but "close" does not reset the eof
   flag, and "fclose" does... */

   fclose ( 8 );
   fclose ( 9 );
}

BYTE diskerr( when )
char *when;
/* check for disk error, and, if so,
close all disk channels, print error
msg and return error number */
{
   register BYTE err;

   if ( err = ferror() )
   {
      SETCOMP;
      dclose ();
      printf ( "%cError #%d %s\n", CLRS, err, when );
      stall ();
   }
   return err;
}

VOID instr1()
/* print the instructions which go on
the screen requesting track and sector
number */
{
   SETCOMP;
   printf ( "%c\n\n              Full Screen\n\n", CLRS );
   printf ( "  D i s k   S e c t o r   E d i t o r\n\n" );
   printf ( "          Raymond L. Zarling\n\n\n\n" );
   printf ( "    Enter Track and Sector number\n" );
   printf ( "       in Decimal.\n\n" );
   printf ( "    Mode is A for ASCII or H for\n" );
   printf ( "       Hexadecimal Display.\n\n" );
   printf ( "    Enter Track 0 to Quit.%s\n\n\n" );
}

VOID instr2()
/* print the instructions which go on
the sector display screen. */
{
   printf ( "\np (or r)   print sector (r = rs232)\n" );
   printf ( ":adr data  edit data\n" );
   printf ( "- t s      write this data to new t, s\n" );
   printf ( "a (or h)   redisplay in ASCII (or hex)\n" );
   printf ( "x          don't write changes to disk\n" );
   printf ( "w          write changes to disk" );
}

VOID dsect()
/* display the track and sector numbers,
and all the data in the current sector */
{
   register BYTE i;

   SETCOMP;
   if ( outf != stdout ) lineout ( "" );
   sprintf ( image, "--  Track %d Sector %d", tr, se );
   lineout ( image );
   lineout ( "" );

   i = 0;
   do
   {
      ldisp ( i );
      i += 16;
   }  while ( i );
}

VOID display ()
/* generate the entire sector display,
with instructions */
{
   putchar ( CLRS );
   dsect ();
   instr2 ();
}

BOOL badts ()
/* return TRUE if tr, se represent an
illegal track and sector combination */
{
static BYTE tsize[]
  = { 0, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20,
     20, 20, 20, 20, 20, 20, 20, 18, 18, 18,
     18, 18, 18, 18, 17, 17, 17, 17, 17, 17,
     16, 16, 16, 16, 16};

   return ( tr < 1 || tr > 35
         || tsize[ tr ] < se );
}

BOOL device ( d, s )
unsigned /* BYTE */ d;
char *s;
/* switch the current output device to
device d ( = 2, 3, or 4 ).  s is the
string to use in opening the device.
use file outf as the file descriptor;
outf=stdout only if d == 3. */
{
   while ( *ENABL & 3 ); /* wait 'till 232 done */
   close ( 5 );
   rs232 = ( d == 2 );
   if ( d == 3 )
      outf = stdout;
   else if ( open (5, d, 7, s) || d <= 3 )
      {
         outf = 5;
         return TRUE;
      }
      else
      {
         device ( 3 );
         SETCOMP;
         printf ( "%cCan't Open Device %d\n", CLRS, d );
         stall ();
         display ();
         return FALSE;
      }
}

VOID stall ()
/* wait for the user to respond */
{
   printf ( "Press RETURN to continue" );
   while ( getchar () != '\n');
}

VOID lineout ( s )
char *s;
/* output the string s to the current
device, followed by \n. */
{
   register char c;

   while ( c = *s++ )
   {
      if ( rs232 )
         c = toascii ( c );
      putc ( c, outf );
   }
   putc ( '\n', outf );
   if ( rs232 )
      putc ( '\012', outf );
}

char toascii ( ch )
char ch;
/* convert ch to true ascii */
{
   if ( isupper(ch) )
      return ( ch & 0x7f );
   if ( islower(ch) )
      return ( ch | 0x20 );
   return ch;
}

VOID tone ()
/* sound an error tone */
{
   BYTE time;

   *(unsigned *)SID = 0x0400;
   *(unsigned *)(SID + 5) = 0xf00f;
   *(char *)(SID + 4) = 0x21;
   *(char *)(SID + 24) = 0x0f;
   time = *JCLOCK + 15;
   while ( time != *JCLOCK );
   *(char *)(SID + 4) = 0;
}
